﻿using System.Collections.Generic;
using UnityEngine;

namespace KerbalMass
{
    /// <summary>
    /// This class handles loading of the settings file and provides data for other classes and objects.
    /// </summary>
    [KSPAddon(KSPAddon.Startup.Instantly,false)]
    public class KerbalMassConfigManager : MonoBehaviour
    {
        /// <summary>
        /// Mass of a Kerbal in metric tons.
        /// </summary>
        /// <remarks>
        /// Got the default number (0.09375) from the wiki. (http://wiki.kerbalspaceprogram.com/wiki/Kerbal)
        /// You can also get the mass by calculating KerbalEVA.initialMass * KerbalEVA.massMultiplier (3.125 tons * 0.03).
        /// </remarks>
        public static float Kerbal_Mass = 0f;

        /// <summary>
        /// If set to true then it's assumed that a part's default mass already includes the mass of a full crew.
        /// </summary>
        public static bool DefaultPartMassIncludesKerbalMass = false;

        /// <summary>
        /// Static constructor! What did you expect me to write here?
        /// </summary>
        static KerbalMassConfigManager()
        {
            ReadConfig();
        }

        /// <summary>
        /// Reads settings.
        /// It also throws a bunch of error messages if something goes wrong and then creates/overrides the settings file.
        /// 
        /// ToDo: Refactor!
        /// 
        /// </summary>
        private static void ReadConfig()
        {
            ConfigNode[] allSettingsNodes = GameDatabase.Instance.GetConfigNodes("KerbalMass_Settings");
            ConfigNode settings = null;
            bool needToSaveConfig = false;

            if (allSettingsNodes.Length == 1)
            {
                settings = allSettingsNodes[0];

                if (settings != null)
                {
                    // KerbalMass
                    if (GetKerbalMassValue(settings, out Kerbal_Mass) == false)
                    {
                        Debug.LogError("[KerbalMass] There was a problem reading the field \"KerbalMass\" or it's value.\nUsing default value.");
                        
                        Kerbal_Mass = 0.09375f;
                        settings.RemoveValuesStartWith("KerbalMass");
                        settings.AddValue("KerbalMass", Kerbal_Mass);
                        needToSaveConfig = true;
                    }

                    // DefaultPartMassIncludesKerbalMass
                    if (GetDefaultPartMassIncludesKerbalMass(settings, out DefaultPartMassIncludesKerbalMass) == false)
                    {
                        Debug.LogError("[KerbalMass] There was a problem reading the field \"DefaultPartMassIncludesKerbalMass\" or it's value.\nUsing default value.");

                        DefaultPartMassIncludesKerbalMass = false;
                        settings.RemoveValuesStartWith("DefaultPartMassIncludesKerbalMass");
                        settings.AddValue("DefaultPartMassIncludesKerbalMass", DefaultPartMassIncludesKerbalMass);
                        needToSaveConfig = true;
                    }
                }
            }
            else
            {
                Kerbal_Mass = 0.09375f;
                DefaultPartMassIncludesKerbalMass = false;
                
                settings = new ConfigNode();
                settings.AddValue("KerbalMass", Kerbal_Mass);
                settings.AddValue("DefaultPartMassIncludesKerbalMass", DefaultPartMassIncludesKerbalMass);                
            }

            if (needToSaveConfig)
            {
                settings.Save("\\GameData\\KerbalMass\\KerbalMass_Settings.cfg");
                Debug.Log("[KerbalMass] The settings file seems to be missing or is damaged. Created a new one.");
            }

            Debug.Log("[KerbalMass] A Kerbal weigths " + (Kerbal_Mass * 1000f) + " kg.");
            Debug.Log("[KerbalMass] A part's default mass " + (DefaultPartMassIncludesKerbalMass ? "doesn't include" : "includes") + " Kerbal masses.");
        }

        /// <summary>
        /// Reads the Kerbal's mass from the ConfigNode object.
        /// </summary>
        /// <param name="configNode">ConfigNode object</param>
        /// <param name="value">Kerbal's mass in game units (metric tons), reset to 0 if the was an error</param>
        /// <returns>true if read was succesful; false if unsuccesful or value is invalid, value is reset to 0</returns>
        private static bool GetKerbalMassValue(ConfigNode configNode, out float value)
        {
            string kerbalMassString = configNode.GetValue("KerbalMass");

            if (!string.IsNullOrEmpty(kerbalMassString))
                if (float.TryParse(kerbalMassString, out value))
                    if (value > 0f)
                        return true;
                    else
                        Debug.LogError("[KerbalMass] Value is negative or 0!");

                else
                    Debug.LogError("[KerbalMass] Couldn't convert value to a number!");
            
            else
                Debug.LogError("[KerbalMass] Value doesn't exist or is invalid!");

            value = 0f;
            return false;
        }

        /// <summary>
        /// Reads DefaultPartMassIncludesKerbalMass's value from the ConfigNode.
        /// </summary>
        /// <param name="configNode">settings file/ConfigNode object</param>
        /// <param name="value">DefaultPartMassIncludesKerbalMass's value</param>
        /// <returns>true if read was succesful, reset to false if there was an error</returns>
        private static bool GetDefaultPartMassIncludesKerbalMass(ConfigNode configNode, out bool value)
        {
            string defaultPartMassIncludesKerbalMass = configNode.GetValue("DefaultPartMassIncludesKerbalMass");

            if (!string.IsNullOrEmpty(defaultPartMassIncludesKerbalMass))
                if (bool.TryParse(defaultPartMassIncludesKerbalMass, out value))
                    return true;
                else
                    Debug.LogError("[KerbalMass] Couldn't convert value to a bool!");

            else
                Debug.LogError("[KerbalMass] Value doesn't exist or is invalid!");

            value = false;
            return false;
        }
    }
}
